// JavaScript pour la page de spectacle

document.addEventListener("DOMContentLoaded", () => {
  initPerformancePage()
  initBookingButtons()
  initFavoriteToggle()
})

function initPerformancePage() {
  console.log("Page spectacle initialisée")

  // Charger les avis
  loadReviews()

  // Initialiser le carousel d'images si présent
  initImageCarousel()
}

function initBookingButtons() {
  const bookingButtons = document.querySelectorAll('a[href*="reservation.php"]')

  bookingButtons.forEach((button) => {
    button.addEventListener("click", function (e) {
      // Vérifier la disponibilité avant de rediriger
      const scheduleId = new URL(this.href).searchParams.get("schedule_id")
      if (scheduleId) {
        checkAvailability(scheduleId, this.href)
        e.preventDefault()
      }
    })
  })
}

function checkAvailability(scheduleId, redirectUrl) {
  fetch(`api/check-availability.php?schedule_id=${scheduleId}`)
    .then((response) => response.json())
    .then((data) => {
      if (data.available) {
        window.location.href = redirectUrl
      } else {
        showNotification("Désolé, cette séance est complète", "error")
        // Mettre à jour l'affichage
        updateScheduleStatus(scheduleId, "full")
      }
    })
    .catch((error) => {
      console.error("Erreur de vérification:", error)
      showNotification("Erreur de vérification de disponibilité", "error")
    })
}

function updateScheduleStatus(scheduleId, status) {
  const scheduleItem = document.querySelector(`[data-schedule-id="${scheduleId}"]`)
  if (scheduleItem) {
    const button = scheduleItem.querySelector(".btn")
    if (status === "full") {
      button.textContent = "Complet"
      button.classList.add("disabled")
      button.style.pointerEvents = "none"
    }
  }
}

function initFavoriteToggle() {
  const favoriteButton = document.querySelector(".btn-favorite")

  if (favoriteButton) {
    favoriteButton.addEventListener("click", function () {
      const performanceId = this.dataset.performanceId
      toggleFavorite(performanceId)
    })
  }
}

function loadReviews() {
  const performanceId = new URLSearchParams(window.location.search).get("id")

  if (performanceId) {
    fetch(`api/reviews.php?performance_id=${performanceId}`)
      .then((response) => response.json())
      .then((data) => {
        displayReviews(data)
      })
      .catch((error) => {
        console.error("Erreur de chargement des avis:", error)
      })
  }
}

function displayReviews(reviews) {
  const reviewsContainer = document.getElementById("reviews-container")

  if (!reviewsContainer) return

  reviewsContainer.innerHTML = ""

  reviews.forEach((review) => {
    const reviewElement = createReviewElement(review)
    reviewsContainer.appendChild(reviewElement)
  })
}

function createReviewElement(review) {
  const reviewDiv = document.createElement("div")
  reviewDiv.className = "review-item"

  const stars = "★".repeat(review.rating) + "☆".repeat(5 - review.rating)

  reviewDiv.innerHTML = `
        <div class="review-header">
            <div class="review-author">${review.author_name}</div>
            <div class="review-rating">${stars}</div>
            <div class="review-date">${formatDate(review.created_at)}</div>
        </div>
        <div class="review-comment">${review.comment}</div>
    `

  return reviewDiv
}

function formatDate(dateString) {
  const date = new Date(dateString)
  return date.toLocaleDateString("fr-FR", {
    year: "numeric",
    month: "long",
    day: "numeric",
  })
}

function initImageCarousel() {
  const carousel = document.querySelector(".image-carousel")

  if (carousel) {
    let currentIndex = 0
    const images = carousel.querySelectorAll("img")
    const totalImages = images.length

    const prevButton = carousel.querySelector(".carousel-prev")
    const nextButton = carousel.querySelector(".carousel-next")

    if (prevButton) {
      prevButton.addEventListener("click", () => {
        currentIndex = (currentIndex - 1 + totalImages) % totalImages
        updateCarousel()
      })
    }

    if (nextButton) {
      nextButton.addEventListener("click", () => {
        currentIndex = (currentIndex + 1) % totalImages
        updateCarousel()
      })
    }

    function updateCarousel() {
      const offset = -currentIndex * 100
      carousel.querySelector(".carousel-track").style.transform = `translateX(${offset}%)`
    }
  }
}

// Mock functions to resolve the "noUndeclaredVariables" errors.
// In a real application, these would be defined elsewhere or imported.
function showNotification(message, type) {
  console.log(`Notification: ${message} (type: ${type})`)
}

function toggleFavorite(performanceId) {
  console.log(`Toggling favorite for performance ID: ${performanceId}`)
}
