-- Base de données pour Nouvel Acte
CREATE DATABASE IF NOT EXISTS nouvel_acte;
USE nouvel_acte;

-- Table des rôles
CREATE TABLE roles (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name ENUM('admin', 'client') NOT NULL
);

-- Table des utilisateurs
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    display_name VARCHAR(100) NOT NULL,
    avatar_url VARCHAR(255),
    role_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (role_id) REFERENCES roles(id)
);

-- Table des catégories
CREATE TABLE categories (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    slug VARCHAR(100) UNIQUE NOT NULL
);

-- Table des salles
CREATE TABLE halls (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    capacity INT NOT NULL
);

-- Table des spectacles
CREATE TABLE performances (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL,
    slug VARCHAR(255) UNIQUE NOT NULL,
    description TEXT,
    duration INT NOT NULL,
    min_age TINYINT DEFAULT 0,
    category_id INT,
    poster_url VARCHAR(255),
    status ENUM('draft', 'online', 'archived') DEFAULT 'draft',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES categories(id)
);

-- Table des séances
CREATE TABLE schedules (
    id INT PRIMARY KEY AUTO_INCREMENT,
    performance_id INT NOT NULL,
    hall_id INT NOT NULL,
    show_datetime DATETIME NOT NULL,
    price DECIMAL(8,2) NOT NULL,
    available_seats INT NOT NULL,
    status ENUM('open', 'full', 'cancelled') DEFAULT 'open',
    FOREIGN KEY (performance_id) REFERENCES performances(id),
    FOREIGN KEY (hall_id) REFERENCES halls(id)
);

-- Table des réservations
CREATE TABLE reservations (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    total_price DECIMAL(8,2) NOT NULL,
    booked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    status ENUM('pending', 'paid', 'cancelled', 'refunded') DEFAULT 'pending',
    payment_ref VARCHAR(100),
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Table des éléments de réservation
CREATE TABLE reservation_items (
    id INT PRIMARY KEY AUTO_INCREMENT,
    reservation_id INT NOT NULL,
    schedule_id INT NOT NULL,
    seat_number VARCHAR(10),
    price DECIMAL(8,2) NOT NULL,
    FOREIGN KEY (reservation_id) REFERENCES reservations(id),
    FOREIGN KEY (schedule_id) REFERENCES schedules(id)
);

-- Table des avis
CREATE TABLE reviews (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    performance_id INT NOT NULL,
    rating TINYINT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    comment TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    is_approved BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (performance_id) REFERENCES performances(id)
);

-- Table des favoris
CREATE TABLE favorites (
    user_id INT NOT NULL,
    performance_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (user_id, performance_id),
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (performance_id) REFERENCES performances(id)
);

-- Table des amis
CREATE TABLE friends (
    user_id INT NOT NULL,
    friend_id INT NOT NULL,
    status ENUM('requested', 'accepted', 'blocked') DEFAULT 'requested',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_friendship (user_id, friend_id),
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (friend_id) REFERENCES users(id)
);

-- Table des groupes
CREATE TABLE groups (
    id INT PRIMARY KEY AUTO_INCREMENT,
    owner_id INT NOT NULL,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (owner_id) REFERENCES users(id)
);

-- Table des membres de groupes
CREATE TABLE group_members (
    group_id INT NOT NULL,
    user_id INT NOT NULL,
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (group_id, user_id),
    FOREIGN KEY (group_id) REFERENCES groups(id),
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Table des logs
CREATE TABLE logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    action ENUM('create', 'update', 'delete', 'login', 'logout', 'booking', 'payment') NOT NULL,
    entity VARCHAR(50),
    entity_id INT,
    details TEXT,
    ip_address VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Insertion des données de base
INSERT INTO roles (name) VALUES ('admin'), ('client');

INSERT INTO categories (name, slug) VALUES 
('Comédie', 'comedie'),
('Drame', 'drame'),
('Musical', 'musical'),
('Classique', 'classique'),
('Contemporain', 'contemporain'),
('Jeune Public', 'jeune-public');

INSERT INTO halls (name, capacity) VALUES 
('Grande Salle', 300),
('Petite Salle', 150),
('Studio', 80);
