<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: identification.php');
    exit();
}

$performance_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$performance_id) {
    header('Location: dashboard.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Récupérer les détails du spectacle
$query = "SELECT p.*, c.name as category_name, 
          AVG(r.rating) as avg_rating, COUNT(r.id) as review_count
          FROM performances p 
          LEFT JOIN categories c ON p.category_id = c.id
          LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
          WHERE p.id = :id AND p.status = 'online'
          GROUP BY p.id";
$stmt = $db->prepare($query);
$stmt->bindParam(':id', $performance_id);
$stmt->execute();
$performance = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$performance) {
    header('Location: dashboard.php');
    exit();
}

// Récupérer les prochaines séances
$schedule_query = "SELECT s.*, h.name as hall_name 
                   FROM schedules s
                   JOIN halls h ON s.hall_id = h.id
                   WHERE s.performance_id = :id AND s.show_datetime > NOW() AND s.status = 'open'
                   ORDER BY s.show_datetime ASC";
$schedule_stmt = $db->prepare($schedule_query);
$schedule_stmt->bindParam(':id', $performance_id);
$schedule_stmt->execute();
$schedules = $schedule_stmt->fetchAll(PDO::FETCH_ASSOC);

// Vérifier si c'est dans les favoris
$fav_query = "SELECT 1 FROM favorites WHERE user_id = :user_id AND performance_id = :performance_id";
$fav_stmt = $db->prepare($fav_query);
$fav_stmt->bindParam(':user_id', $_SESSION['user_id']);
$fav_stmt->bindParam(':performance_id', $performance_id);
$fav_stmt->execute();
$is_favorite = $fav_stmt->rowCount() > 0;
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($performance['title']); ?> - Nouvel Acte</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="main performance-page">
            <!-- Bannière du spectacle -->
            <div class="performance-banner">
                <div class="banner-bg" style="background-image: url('<?php echo $performance['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>')"></div>
                <div class="banner-content">
                    <h1><?php echo htmlspecialchars($performance['title']); ?></h1>
                    <p><?php echo htmlspecialchars($performance['description']); ?></p>
                    
                    <!-- Note -->
                    <div class="rating">
                        <div class="stars">
                            <?php 
                            $rating = round($performance['avg_rating'] ?: 0);
                            for ($i = 1; $i <= 5; $i++): ?>
                                <span class="<?php echo $i <= $rating ? 'filled' : 'empty'; ?>">★</span>
                            <?php endfor; ?>
                        </div>
                        <a href="avis.php?performance_id=<?php echo $performance_id; ?>">Met ton avis personnel</a>
                    </div>
                    
                    <!-- Actions -->
                    <div class="actions">
                        <a href="reservation.php?performance_id=<?php echo $performance_id; ?>" class="btn btn-primary">
                            Réserve ta place
                        </a>
                        <button class="btn btn-secondary" onclick="showTrailer(<?php echo $performance_id; ?>)">
                            Bande annonce
                        </button>
                        <button class="btn-favorite <?php echo $is_favorite ? 'active' : ''; ?>" 
                                onclick="toggleFavorite(<?php echo $performance_id; ?>)">
                            ♥
                        </button>
                    </div>
                </div>
            </div>

            <!-- Informations détaillées -->
            <div class="performance-details">
                <div class="details-main">
                    <h2>Informations</h2>
                    <div class="info-grid">
                        <div class="info-item">
                            <strong>Durée:</strong> <?php echo $performance['duration']; ?> minutes
                        </div>
                        <div class="info-item">
                            <strong>Âge minimum:</strong> <?php echo $performance['min_age']; ?> ans
                        </div>
                        <div class="info-item">
                            <strong>Catégorie:</strong> <?php echo htmlspecialchars($performance['category_name']); ?>
                        </div>
                        <div class="info-item">
                            <strong>Note moyenne:</strong> 
                            <?php echo number_format($performance['avg_rating'] ?: 0, 1); ?>/5 
                            (<?php echo $performance['review_count']; ?> avis)
                        </div>
                    </div>
                </div>

                <div class="details-sidebar">
                    <h3>Prochaines représentations</h3>
                    <div class="schedules-list">
                        <?php foreach ($schedules as $schedule): ?>
                            <div class="schedule-item">
                                <div class="schedule-info">
                                    <div class="schedule-date">
                                        <?php echo date('d/m/Y à H:i', strtotime($schedule['show_datetime'])); ?>
                                    </div>
                                    <div class="schedule-hall"><?php echo htmlspecialchars($schedule['hall_name']); ?></div>
                                    <div class="schedule-price"><?php echo number_format($schedule['price'], 2); ?>€</div>
                                </div>
                                <div class="schedule-actions">
                                    <span class="seats-available"><?php echo $schedule['available_seats']; ?> places</span>
                                    <a href="reservation.php?schedule_id=<?php echo $schedule['id']; ?>" class="btn btn-small">
                                        Réserver
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script src="assets/js/performance.js"></script>
</body>
</html>
