<?php
session_start();
require_once '../config/database.php';

if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header('Location: ../identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

$message = '';
$action = $_GET['action'] ?? 'list';
$category_id = $_GET['id'] ?? 0;

// Traitement des actions
if ($_POST) {
    if ($action === 'add' || $action === 'edit') {
        $name = trim($_POST['name']);
        $slug = strtolower(str_replace(' ', '-', $name));
        
        if ($action === 'add') {
            $query = "INSERT INTO categories (name, slug) VALUES (:name, :slug)";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':name', $name);
            $stmt->bindParam(':slug', $slug);
            
            if ($stmt->execute()) {
                $message = 'Catégorie ajoutée avec succès';
                $action = 'list';
            }
        } else {
            $query = "UPDATE categories SET name = :name, slug = :slug WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':name', $name);
            $stmt->bindParam(':slug', $slug);
            $stmt->bindParam(':id', $category_id);
            
            if ($stmt->execute()) {
                $message = 'Catégorie modifiée avec succès';
                $action = 'list';
            }
        }
    }
}

// Suppression
if ($action === 'delete' && $category_id) {
    // Vérifier s'il y a des spectacles dans cette catégorie
    $check_query = "SELECT COUNT(*) as count FROM performances WHERE category_id = :id";
    $check_stmt = $db->prepare($check_query);
    $check_stmt->bindParam(':id', $category_id);
    $check_stmt->execute();
    $check_result = $check_stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($check_result['count'] > 0) {
        $message = 'Impossible de supprimer cette catégorie car elle contient des spectacles';
    } else {
        $delete_query = "DELETE FROM categories WHERE id = :id";
        $delete_stmt = $db->prepare($delete_query);
        $delete_stmt->bindParam(':id', $category_id);
        if ($delete_stmt->execute()) {
            $message = 'Catégorie supprimée avec succès';
        }
    }
    $action = 'list';
}

// Récupération des données
if ($action === 'list') {
    $query = "SELECT c.*, COUNT(p.id) as performances_count 
              FROM categories c 
              LEFT JOIN performances p ON c.id = p.category_id 
              GROUP BY c.id 
              ORDER BY c.name";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

if ($action === 'edit' && $category_id) {
    $query = "SELECT * FROM categories WHERE id = :id";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':id', $category_id);
    $stmt->execute();
    $category = $stmt->fetch(PDO::FETCH_ASSOC);
}
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des catégories - Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="admin-main">
            <header class="admin-header">
                <h1>Gestion des catégories</h1>
                <?php if ($action === 'list'): ?>
                <a href="?action=add" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Ajouter une catégorie
                </a>
                <?php endif; ?>
            </header>

            <div class="admin-content">
                <?php if ($message): ?>
                <div class="alert <?php echo strpos($message, 'Impossible') !== false ? 'alert-danger' : 'alert-success'; ?>">
                    <i class="fas fa-<?php echo strpos($message, 'Impossible') !== false ? 'exclamation-triangle' : 'check-circle'; ?>"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <?php endif; ?>

                <?php if ($action === 'list'): ?>
                <!-- Liste des catégories -->
                <div class="admin-card">
                    <div class="table-responsive">
                        <table class="admin-table">
                            <thead>
                                <tr>
                                    <th>Nom</th>
                                    <th>Slug</th>
                                    <th>Spectacles</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($categories as $cat): ?>
                                <tr>
                                    <td><strong><?php echo htmlspecialchars($cat['name']); ?></strong></td>
                                    <td><code><?php echo htmlspecialchars($cat['slug']); ?></code></td>
                                    <td><?php echo $cat['performances_count']; ?> spectacle(s)</td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="?action=edit&id=<?php echo $cat['id']; ?>" class="btn btn-sm btn-outline">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <?php if ($cat['performances_count'] == 0): ?>
                                            <a href="?action=delete&id=<?php echo $cat['id']; ?>" 
                                               class="btn btn-sm btn-danger" 
                                               onclick="return confirm('Êtes-vous sûr de vouloir supprimer cette catégorie ?')">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <?php elseif ($action === 'add' || $action === 'edit'): ?>
                <!-- Formulaire d'ajout/modification -->
                <div class="admin-card">
                    <div class="card-header">
                        <h3><?php echo $action === 'add' ? 'Ajouter' : 'Modifier'; ?> une catégorie</h3>
                        <a href="?action=list" class="btn btn-outline">
                            <i class="fas fa-arrow-left"></i> Retour
                        </a>
                    </div>
                    
                    <form method="POST" class="admin-form">
                        <div class="form-group">
                            <label for="name">Nom de la catégorie *</label>
                            <input type="text" id="name" name="name" required 
                                   value="<?php echo htmlspecialchars($category['name'] ?? ''); ?>"
                                   placeholder="Ex: Comédie, Drame, Musical...">
                        </div>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i>
                                <?php echo $action === 'add' ? 'Ajouter' : 'Modifier'; ?>
                            </button>
                            <a href="?action=list" class="btn btn-outline">Annuler</a>
                        </div>
                    </form>
                </div>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <script src="assets/js/admin.js"></script>
</body>
</html>
