<?php
session_start();
require_once '../config/database.php';

if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'admin') {
    header('Location: ../identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

$message = '';
$action = $_GET['action'] ?? 'list';
$user_id = $_GET['id'] ?? 0;

// Traitement des actions
if ($_POST) {
    if ($action === 'add' || $action === 'edit') {
        $email = trim($_POST['email']);
        $display_name = trim($_POST['display_name']);
        $role_id = (int)$_POST['role_id'];
        
        if ($action === 'add') {
            $password = $_POST['password'];
            $password_hash = password_hash($password, PASSWORD_DEFAULT);
            
            $query = "INSERT INTO users (email, password_hash, display_name, role_id) 
                     VALUES (:email, :password_hash, :display_name, :role_id)";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':password_hash', $password_hash);
            $stmt->bindParam(':display_name', $display_name);
            $stmt->bindParam(':role_id', $role_id);
            
            if ($stmt->execute()) {
                $message = 'Utilisateur ajouté avec succès';
                $action = 'list';
            }
        } else {
            $query = "UPDATE users SET email = :email, display_name = :display_name, role_id = :role_id";
            $params = [':email' => $email, ':display_name' => $display_name, ':role_id' => $role_id, ':id' => $user_id];
            
            if (!empty($_POST['password'])) {
                $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);
                $query .= ", password_hash = :password_hash";
                $params[':password_hash'] = $password_hash;
            }
            
            $query .= " WHERE id = :id";
            $stmt = $db->prepare($query);
            
            foreach ($params as $key => $value) {
                $stmt->bindValue($key, $value);
            }
            
            if ($stmt->execute()) {
                $message = 'Utilisateur modifié avec succès';
                $action = 'list';
            }
        }
    }
}

// Suppression
if ($action === 'delete' && $user_id) {
    $delete_query = "DELETE FROM users WHERE id = :id AND id != :current_user";
    $delete_stmt = $db->prepare($delete_query);
    $delete_stmt->bindParam(':id', $user_id);
    $delete_stmt->bindParam(':current_user', $_SESSION['user_id']);
    if ($delete_stmt->execute()) {
        $message = 'Utilisateur supprimé avec succès';
    }
    $action = 'list';
}

// Récupération des données
if ($action === 'list') {
    $query = "SELECT u.*, r.name as role_name,
              COUNT(DISTINCT res.id) as reservations_count,
              COUNT(DISTINCT rev.id) as reviews_count
              FROM users u 
              LEFT JOIN roles r ON u.role_id = r.id
              LEFT JOIN reservations res ON u.id = res.user_id
              LEFT JOIN reviews rev ON u.id = rev.user_id
              GROUP BY u.id
              ORDER BY u.created_at DESC";
    $stmt = $db->prepare($query);
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

if ($action === 'edit' && $user_id) {
    $query = "SELECT * FROM users WHERE id = :id";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':id', $user_id);
    $stmt->execute();
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
}

// Récupération des rôles
$roles_query = "SELECT * FROM roles ORDER BY name";
$roles_stmt = $db->prepare($roles_query);
$roles_stmt->execute();
$roles = $roles_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des utilisateurs - Admin</title>
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="admin-main">
            <header class="admin-header">
                <h1>Gestion des utilisateurs</h1>
                <?php if ($action === 'list'): ?>
                <a href="?action=add" class="btn btn-primary">
                    <i class="fas fa-plus"></i> Ajouter un utilisateur
                </a>
                <?php endif; ?>
            </header>

            <div class="admin-content">
                <?php if ($message): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <?php endif; ?>

                <?php if ($action === 'list'): ?>
                <!-- Liste des utilisateurs -->
                <div class="admin-card">
                    <div class="table-responsive">
                        <table class="admin-table">
                            <thead>
                                <tr>
                                    <th>Utilisateur</th>
                                    <th>Rôle</th>
                                    <th>Réservations</th>
                                    <th>Avis</th>
                                    <th>Inscription</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($users as $user): ?>
                                <tr>
                                    <td>
                                        <div class="user-info">
                                            <strong><?php echo htmlspecialchars($user['display_name']); ?></strong>
                                            <small><?php echo htmlspecialchars($user['email']); ?></small>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="status status-<?php echo $user['role_name']; ?>">
                                            <?php echo ucfirst($user['role_name']); ?>
                                        </span>
                                    </td>
                                    <td><?php echo $user['reservations_count']; ?></td>
                                    <td><?php echo $user['reviews_count']; ?></td>
                                    <td><?php echo date('d/m/Y', strtotime($user['created_at'])); ?></td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="?action=edit&id=<?php echo $user['id']; ?>" class="btn btn-sm btn-outline">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                            <a href="?action=delete&id=<?php echo $user['id']; ?>" 
                                               class="btn btn-sm btn-danger" 
                                               onclick="return confirm('Êtes-vous sûr de vouloir supprimer cet utilisateur ?')">
                                                <i class="fas fa-trash"></i>
                                            </a>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <?php elseif ($action === 'add' || $action === 'edit'): ?>
                <!-- Formulaire d'ajout/modification -->
                <div class="admin-card">
                    <div class="card-header">
                        <h3><?php echo $action === 'add' ? 'Ajouter' : 'Modifier'; ?> un utilisateur</h3>
                        <a href="?action=list" class="btn btn-outline">
                            <i class="fas fa-arrow-left"></i> Retour
                        </a>
                    </div>
                    
                    <form method="POST" class="admin-form">
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="display_name">Nom complet *</label>
                                <input type="text" id="display_name" name="display_name" required 
                                       value="<?php echo htmlspecialchars($user['display_name'] ?? ''); ?>">
                            </div>

                            <div class="form-group">
                                <label for="email">Email *</label>
                                <input type="email" id="email" name="email" required 
                                       value="<?php echo htmlspecialchars($user['email'] ?? ''); ?>">
                            </div>

                            <div class="form-group">
                                <label for="role_id">Rôle *</label>
                                <select id="role_id" name="role_id" required>
                                    <?php foreach ($roles as $role): ?>
                                    <option value="<?php echo $role['id']; ?>" 
                                            <?php echo ($user['role_id'] ?? '') == $role['id'] ? 'selected' : ''; ?>>
                                        <?php echo ucfirst($role['name']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="form-group">
                                <label for="password">
                                    Mot de passe <?php echo $action === 'add' ? '*' : '(laisser vide pour ne pas changer)'; ?>
                                </label>
                                <input type="password" id="password" name="password" 
                                       <?php echo $action === 'add' ? 'required' : ''; ?>>
                            </div>
                        </div>

                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i>
                                <?php echo $action === 'add' ? 'Ajouter' : 'Modifier'; ?>
                            </button>
                            <a href="?action=list" class="btn btn-outline">Annuler</a>
                        </div>
                    </form>
                </div>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <script src="assets/js/admin.js"></script>
</body>
</html>
