<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Récupérer les favoris de l'utilisateur
$favorites_query = "SELECT p.*, c.name as category_name, 
                   AVG(r.rating) as avg_rating, COUNT(r.id) as review_count,
                   MIN(s.show_datetime) as next_show, MIN(s.price) as min_price
                   FROM favorites f
                   JOIN performances p ON f.performance_id = p.id
                   LEFT JOIN categories c ON p.category_id = c.id
                   LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
                   LEFT JOIN schedules s ON p.id = s.performance_id AND s.show_datetime > NOW() AND s.status = 'open'
                   WHERE f.user_id = :user_id AND p.status = 'online'
                   GROUP BY p.id
                   ORDER BY f.created_at DESC";
$favorites_stmt = $db->prepare($favorites_query);
$favorites_stmt->bindParam(':user_id', $_SESSION['user_id']);
$favorites_stmt->execute();
$favorites = $favorites_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les recommandations basées sur les favoris
$recommendations_query = "SELECT DISTINCT p.*, c.name as category_name, 
                         AVG(r.rating) as avg_rating, COUNT(r.id) as review_count
                         FROM performances p
                         LEFT JOIN categories c ON p.category_id = c.id
                         LEFT JOIN reviews r ON p.id = r.performance_id AND r.is_approved = 1
                         WHERE p.category_id IN (
                             SELECT DISTINCT p2.category_id 
                             FROM favorites f2 
                             JOIN performances p2 ON f2.performance_id = p2.id 
                             WHERE f2.user_id = :user_id
                         )
                         AND p.id NOT IN (
                             SELECT performance_id FROM favorites WHERE user_id = :user_id
                         )
                         AND p.status = 'online'
                         GROUP BY p.id
                         ORDER BY avg_rating DESC
                         LIMIT 4";
$recommendations_stmt = $db->prepare($recommendations_query);
$recommendations_stmt->bindParam(':user_id', $_SESSION['user_id']);
$recommendations_stmt->execute();
$recommendations = $recommendations_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mes Favoris - Nouvel Acte</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>

        <header class="header">
            <div class="search-box">
                <div class="icon search-icon"></div>
                <input type="text" placeholder="Rechercher dans mes favoris" id="searchInput">
                <div class="icon filter-icon"></div>
            </div>
            <div class="actions">
                <div class="icon-btn notification-btn" title="Notifications"></div>
            </div>
        </header>

        <main class="main">
            <div class="page-header">
                <h2>❤️ Mes Favoris</h2>
                <p><?php echo count($favorites); ?> spectacle(s) dans vos favoris</p>
            </div>

            <?php if (empty($favorites)): ?>
            <div class="empty-state">
                <div class="empty-icon">💔</div>
                <h3>Aucun favori pour le moment</h3>
                <p>Découvrez des spectacles et ajoutez-les à vos favoris pour les retrouver ici.</p>
                <a href="dashboard.php" class="btn btn-primary">Découvrir des spectacles</a>
            </div>
            <?php else: ?>
            
            <!-- Filtres -->
            <div class="filters">
                <button class="filter-btn active" data-filter="all">Tous</button>
                <button class="filter-btn" data-filter="available">Disponibles</button>
                <button class="filter-btn" data-filter="upcoming">Prochainement</button>
            </div>

            <!-- Liste des favoris -->
            <div class="favorites-grid">
                <?php foreach ($favorites as $performance): ?>
                <div class="favorite-card" data-category="<?php echo $performance['category_name']; ?>">
                    <div class="card-image" style="background-image: url('<?php echo $performance['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>')">
                        <button class="btn-favorite active" onclick="toggleFavorite(<?php echo $performance['id']; ?>)">
                            ♥
                        </button>
                        <?php if ($performance['next_show']): ?>
                        <div class="availability-badge available">Disponible</div>
                        <?php else: ?>
                        <div class="availability-badge unavailable">Indisponible</div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="card-content">
                        <div class="card-header">
                            <h3><?php echo htmlspecialchars($performance['title']); ?></h3>
                            <span class="category"><?php echo htmlspecialchars($performance['category_name']); ?></span>
                        </div>
                        
                        <div class="card-info">
                            <div class="rating">
                                <span class="stars">
                                    <?php 
                                    $rating = round($performance['avg_rating'] ?: 0);
                                    for ($i = 1; $i <= 5; $i++): ?>
                                        <span class="<?php echo $i <= $rating ? 'filled' : 'empty'; ?>">★</span>
                                    <?php endfor; ?>
                                </span>
                                <span class="rating-text">
                                    <?php echo number_format($performance['avg_rating'] ?: 0, 1); ?> 
                                    (<?php echo $performance['review_count']; ?> avis)
                                </span>
                            </div>
                            
                            <?php if ($performance['next_show']): ?>
                            <div class="next-show">
                                <strong>Prochaine séance :</strong>
                                <?php echo date('d/m/Y à H:i', strtotime($performance['next_show'])); ?>
                            </div>
                            <div class="price">
                                À partir de <?php echo number_format($performance['min_price'], 2); ?>€
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="card-actions">
                            <a href="performance.php?id=<?php echo $performance['id']; ?>" class="btn btn-primary">
                                Voir le spectacle
                            </a>
                            <?php if ($performance['next_show']): ?>
                            <a href="reservation.php?performance_id=<?php echo $performance['id']; ?>" class="btn btn-secondary">
                                Réserver
                            </a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>

            <!-- Recommandations -->
            <?php if (!empty($recommendations)): ?>
            <section class="recommendations-section">
                <h3>💡 Vous pourriez aussi aimer</h3>
                <div class="recommendations-grid">
                    <?php foreach ($recommendations as $rec): ?>
                    <div class="recommendation-card" onclick="viewPerformance(<?php echo $rec['id']; ?>)">
                        <div class="rec-image" style="background-image: url('<?php echo $rec['poster_url'] ?: 'assets/images/placeholder.jpg'; ?>')"></div>
                        <div class="rec-info">
                            <h4><?php echo htmlspecialchars($rec['title']); ?></h4>
                            <div class="rec-rating">
                                ★ <?php echo number_format($rec['avg_rating'] ?: 0, 1); ?>
                            </div>
                        </div>
                        <button class="btn-add-favorite" onclick="event.stopPropagation(); toggleFavorite(<?php echo $rec['id']; ?>)">
                            ♡
                        </button>
                    </div>
                    <?php endforeach; ?>
                </div>
            </section>
            <?php endif; ?>
            
            <?php endif; ?>
        </main>

        <aside class="aside">
            <div class="profile">
                <img src="assets/images/avatar-placeholder.jpg" alt="Avatar"/>
                <div class="info">
                    <div><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                    <div>@<?php echo strtolower(str_replace(' ', '', $_SESSION['user_name'])); ?></div>
                </div>
            </div>
            
            <div class="stats-card">
                <h4>📊 Vos statistiques</h4>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count($favorites); ?></span>
                    <span class="stat-label">Favoris</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count(array_filter($favorites, function($f) { return $f['next_show']; })); ?></span>
                    <span class="stat-label">Disponibles</span>
                </div>
            </div>
            
            <button class="btn-all" onclick="location.href='dashboard.php'">Découvrir plus</button>
        </aside>
    </div>

    <script src="assets/js/favorites.js"></script>
</body>
</html>
