<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Traitement des actions
if ($_POST) {
    $action = $_POST['action'];
    $friend_id = (int)$_POST['friend_id'];
    
    if ($action === 'send_request') {
        $query = "INSERT IGNORE INTO friends (user_id, friend_id, status) VALUES (:user_id, :friend_id, 'requested')";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->bindParam(':friend_id', $friend_id);
        $stmt->execute();
    } elseif ($action === 'accept_request') {
        $query = "UPDATE friends SET status = 'accepted' WHERE user_id = :friend_id AND friend_id = :user_id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':friend_id', $friend_id);
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->execute();
        
        // Créer la relation inverse
        $query2 = "INSERT IGNORE INTO friends (user_id, friend_id, status) VALUES (:user_id, :friend_id, 'accepted')";
        $stmt2 = $db->prepare($query2);
        $stmt2->bindParam(':user_id', $_SESSION['user_id']);
        $stmt2->bindParam(':friend_id', $friend_id);
        $stmt2->execute();
    } elseif ($action === 'decline_request') {
        $query = "DELETE FROM friends WHERE user_id = :friend_id AND friend_id = :user_id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':friend_id', $friend_id);
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->execute();
    }
}

// Récupérer les amis acceptés
$friends_query = "SELECT u.*, f.created_at as friendship_date
                  FROM friends f
                  JOIN users u ON f.friend_id = u.id
                  WHERE f.user_id = :user_id AND f.status = 'accepted'
                  ORDER BY f.created_at DESC";
$friends_stmt = $db->prepare($friends_query);
$friends_stmt->bindParam(':user_id', $_SESSION['user_id']);
$friends_stmt->execute();
$friends = $friends_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les demandes en attente
$pending_query = "SELECT u.*, f.created_at as request_date
                  FROM friends f
                  JOIN users u ON f.user_id = u.id
                  WHERE f.friend_id = :user_id AND f.status = 'requested'
                  ORDER BY f.created_at DESC";
$pending_stmt = $db->prepare($pending_query);
$pending_stmt->bindParam(':user_id', $_SESSION['user_id']);
$pending_stmt->execute();
$pending_requests = $pending_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les suggestions d'amis
$suggestions_query = "SELECT u.*, COUNT(DISTINCT common_fav.performance_id) as common_favorites
                     FROM users u
                     LEFT JOIN favorites common_fav ON u.id = common_fav.user_id 
                         AND common_fav.performance_id IN (
                             SELECT performance_id FROM favorites WHERE user_id = :user_id
                         )
                     WHERE u.id != :user_id 
                     AND u.id NOT IN (
                         SELECT friend_id FROM friends WHERE user_id = :user_id
                     )
                     AND u.id NOT IN (
                         SELECT user_id FROM friends WHERE friend_id = :user_id
                     )
                     AND u.role_id = 2
                     GROUP BY u.id
                     ORDER BY common_favorites DESC, RAND()
                     LIMIT 6";
$suggestions_stmt = $db->prepare($suggestions_query);
$suggestions_stmt->bindParam(':user_id', $_SESSION['user_id']);
$suggestions_stmt->execute();
$suggestions = $suggestions_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les activités récentes des amis
$activities_query = "SELECT u.display_name, p.title, r.rating, r.comment, r.created_at
                    FROM reviews r
                    JOIN users u ON r.user_id = u.id
                    JOIN performances p ON r.performance_id = p.id
                    WHERE r.user_id IN (
                        SELECT friend_id FROM friends WHERE user_id = :user_id AND status = 'accepted'
                    )
                    AND r.is_approved = 1
                    ORDER BY r.created_at DESC
                    LIMIT 10";
$activities_stmt = $db->prepare($activities_query);
$activities_stmt->bindParam(':user_id', $_SESSION['user_id']);
$activities_stmt->execute();
$activities = $activities_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mes Amis - Nouvel Acte</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>

        <header class="header">
            <div class="search-box">
                <div class="icon search-icon"></div>
                <input type="text" placeholder="Rechercher des amis" id="searchInput">
                <div class="icon filter-icon"></div>
            </div>
            <div class="actions">
                <div class="icon-btn notification-btn" title="Notifications">
                    <?php if (count($pending_requests) > 0): ?>
                    <span class="notification-badge"><?php echo count($pending_requests); ?></span>
                    <?php endif; ?>
                </div>
            </div>
        </header>

        <main class="main">
            <div class="page-header">
                <h2>👥 Mes Amis</h2>
                <p><?php echo count($friends); ?> ami(s) • <?php echo count($pending_requests); ?> demande(s) en attente</p>
            </div>

            <!-- Demandes en attente -->
            <?php if (!empty($pending_requests)): ?>
            <section class="pending-requests">
                <h3>📬 Demandes d'amitié</h3>
                <div class="requests-list">
                    <?php foreach ($pending_requests as $request): ?>
                    <div class="request-card">
                        <div class="request-avatar">
                            <img src="<?php echo $request['avatar_url'] ?: 'assets/images/avatar-placeholder.jpg'; ?>" 
                                 alt="<?php echo htmlspecialchars($request['display_name']); ?>">
                        </div>
                        <div class="request-info">
                            <h4><?php echo htmlspecialchars($request['display_name']); ?></h4>
                            <p>Demande envoyée le <?php echo date('d/m/Y', strtotime($request['request_date'])); ?></p>
                        </div>
                        <div class="request-actions">
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="accept_request">
                                <input type="hidden" name="friend_id" value="<?php echo $request['id']; ?>">
                                <button type="submit" class="btn btn-primary">Accepter</button>
                            </form>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="decline_request">
                                <input type="hidden" name="friend_id" value="<?php echo $request['id']; ?>">
                                <button type="submit" class="btn btn-outline">Refuser</button>
                            </form>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </section>
            <?php endif; ?>

            <!-- Liste des amis -->
            <section class="friends-list">
                <h3>✅ Mes amis (<?php echo count($friends); ?>)</h3>
                
                <?php if (empty($friends)): ?>
                <div class="empty-state">
                    <div class="empty-icon">👥</div>
                    <h4>Aucun ami pour le moment</h4>
                    <p>Ajoutez des amis pour partager vos découvertes théâtrales !</p>
                </div>
                <?php else: ?>
                
                <div class="friends-grid">
                    <?php foreach ($friends as $friend): ?>
                    <div class="friend-card">
                        <div class="friend-avatar">
                            <img src="<?php echo $friend['avatar_url'] ?: 'assets/images/avatar-placeholder.jpg'; ?>" 
                                 alt="<?php echo htmlspecialchars($friend['display_name']); ?>">
                        </div>
                        <div class="friend-info">
                            <h4><?php echo htmlspecialchars($friend['display_name']); ?></h4>
                            <p>@<?php echo strtolower(str_replace(' ', '', $friend['display_name'])); ?></p>
                            <small>Amis depuis le <?php echo date('d/m/Y', strtotime($friend['friendship_date'])); ?></small>
                        </div>
                        <div class="friend-actions">
                            <button class="btn btn-outline" onclick="viewFriendProfile(<?php echo $friend['id']; ?>)">
                                Profil
                            </button>
                            <button class="btn btn-primary" onclick="inviteToShow(<?php echo $friend['id']; ?>)">
                                Inviter
                            </button>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </section>

            <!-- Suggestions d'amis -->
            <?php if (!empty($suggestions)): ?>
            <section class="friend-suggestions">
                <h3>💡 Suggestions d'amis</h3>
                <div class="suggestions-grid">
                    <?php foreach ($suggestions as $suggestion): ?>
                    <div class="suggestion-card">
                        <div class="suggestion-avatar">
                            <img src="<?php echo $suggestion['avatar_url'] ?: 'assets/images/avatar-placeholder.jpg'; ?>" 
                                 alt="<?php echo htmlspecialchars($suggestion['display_name']); ?>">
                        </div>
                        <div class="suggestion-info">
                            <h4><?php echo htmlspecialchars($suggestion['display_name']); ?></h4>
                            <?php if ($suggestion['common_favorites'] > 0): ?>
                            <p><?php echo $suggestion['common_favorites']; ?> goût(s) en commun</p>
                            <?php else: ?>
                            <p>Nouveau membre</p>
                            <?php endif; ?>
                        </div>
                        <div class="suggestion-actions">
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="send_request">
                                <input type="hidden" name="friend_id" value="<?php echo $suggestion['id']; ?>">
                                <button type="submit" class="btn btn-primary">Ajouter</button>
                            </form>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </section>
            <?php endif; ?>

            <!-- Activités des amis -->
            <?php if (!empty($activities)): ?>
            <section class="friend-activities">
                <h3>🎭 Activités récentes</h3>
                <div class="activities-list">
                    <?php foreach ($activities as $activity): ?>
                    <div class="activity-item">
                        <div class="activity-content">
                            <strong><?php echo htmlspecialchars($activity['display_name']); ?></strong>
                            a donné une note de 
                            <span class="rating">
                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                    <span class="<?php echo $i <= $activity['rating'] ? 'filled' : 'empty'; ?>">★</span>
                                <?php endfor; ?>
                            </span>
                            à <strong><?php echo htmlspecialchars($activity['title']); ?></strong>
                        </div>
                        <?php if ($activity['comment']): ?>
                        <div class="activity-comment">
                            "<?php echo htmlspecialchars(substr($activity['comment'], 0, 100)); ?>..."
                        </div>
                        <?php endif; ?>
                        <div class="activity-date">
                            <?php echo date('d/m/Y à H:i', strtotime($activity['created_at'])); ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </section>
            <?php endif; ?>
        </main>

        <aside class="aside">
            <div class="profile">
                <img src="assets/images/avatar-placeholder.jpg" alt="Avatar"/>
                <div class="info">
                    <div><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                    <div>@<?php echo strtolower(str_replace(' ', '', $_SESSION['user_name'])); ?></div>
                </div>
            </div>
            
            <div class="social-stats">
                <h4>📊 Vos statistiques sociales</h4>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count($friends); ?></span>
                    <span class="stat-label">Amis</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count($pending_requests); ?></span>
                    <span class="stat-label">Demandes</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count($activities); ?></span>
                    <span class="stat-label">Activités</span>
                </div>
            </div>
            
            <div class="quick-invite">
                <h4>⚡ Invitation rapide</h4>
                <p>Invitez vos amis à découvrir un spectacle</p>
                <button class="btn btn-primary" onclick="openInviteModal()">
                    Créer une invitation
                </button>
            </div>
        </aside>
    </div>

    <script src="assets/js/friends.js"></script>
</body>
</html>
