<?php
session_start();
require_once 'config/database.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: identification.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Traitement des actions
if ($_POST) {
    $action = $_POST['action'];
    
    if ($action === 'create_group') {
        $name = trim($_POST['name']);
        $description = trim($_POST['description']);
        
        $query = "INSERT INTO `groups` (owner_id, name, description) VALUES (:owner_id, :name, :description)";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':owner_id', $_SESSION['user_id']);
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':description', $description);
        
        if ($stmt->execute()) {
            $group_id = $db->lastInsertId();
            // Ajouter le créateur comme membre
            $member_query = "INSERT INTO group_members (group_id, user_id) VALUES (:group_id, :user_id)";
            $member_stmt = $db->prepare($member_query);
            $member_stmt->bindParam(':group_id', $group_id);
            $member_stmt->bindParam(':user_id', $_SESSION['user_id']);
            $member_stmt->execute();
        }
    } elseif ($action === 'join_group') {
        $group_id = (int)$_POST['group_id'];
        $query = "INSERT IGNORE INTO group_members (group_id, user_id) VALUES (:group_id, :user_id)";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':group_id', $group_id);
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->execute();
    } elseif ($action === 'leave_group') {
        $group_id = (int)$_POST['group_id'];
        $query = "DELETE FROM group_members WHERE group_id = :group_id AND user_id = :user_id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':group_id', $group_id);
        $stmt->bindParam(':user_id', $_SESSION['user_id']);
        $stmt->execute();
    }
}

// Récupérer les groupes de l'utilisateur
$my_groups_query = "SELECT g.*, u.display_name as owner_name, COUNT(gm.user_id) as member_count,
                   (SELECT COUNT(*) FROM group_members WHERE group_id = g.id AND user_id = :user_id) as is_member
                   FROM `groups` g
                   JOIN users u ON g.owner_id = u.id
                   JOIN group_members gm ON g.id = gm.group_id
                   WHERE g.id IN (SELECT group_id FROM group_members WHERE user_id = :user_id)
                   GROUP BY g.id
                   ORDER BY g.created_at DESC";
$my_groups_stmt = $db->prepare($my_groups_query);
$my_groups_stmt->bindParam(':user_id', $_SESSION['user_id']);
$my_groups_stmt->execute();
$my_groups = $my_groups_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les groupes publics (suggestions)
$public_groups_query = "SELECT g.*, u.display_name as owner_name, COUNT(gm.user_id) as member_count,
                       (SELECT COUNT(*) FROM group_members WHERE group_id = g.id AND user_id = :user_id) as is_member
                       FROM `groups` g
                       JOIN users u ON g.owner_id = u.id
                       LEFT JOIN group_members gm ON g.id = gm.group_id
                       WHERE g.id NOT IN (SELECT group_id FROM group_members WHERE user_id = :user_id)
                       GROUP BY g.id
                       ORDER BY member_count DESC, g.created_at DESC
                       LIMIT 6";
$public_groups_stmt = $db->prepare($public_groups_query);
$public_groups_stmt->bindParam(':user_id', $_SESSION['user_id']);
$public_groups_stmt->execute();
$public_groups = $public_groups_stmt->fetchAll(PDO::FETCH_ASSOC);

// Récupérer les activités récentes des groupes
$group_activities_query = "SELECT g.name as group_name, u.display_name, 'joined' as activity_type, gm.joined_at as activity_date
                          FROM group_members gm
                          JOIN `groups` g ON gm.group_id = g.id
                          JOIN users u ON gm.user_id = u.id
                          WHERE g.id IN (SELECT group_id FROM group_members WHERE user_id = :user_id)
                          AND gm.user_id != :user_id
                          ORDER BY gm.joined_at DESC
                          LIMIT 10";
$activities_stmt = $db->prepare($group_activities_query);
$activities_stmt->bindParam(':user_id', $_SESSION['user_id']);
$activities_stmt->execute();
$group_activities = $activities_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mes Groupes - Nouvel Acte</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="container">
        <?php include 'includes/sidebar.php'; ?>

        <header class="header">
            <div class="search-box">
                <div class="icon search-icon"></div>
                <input type="text" placeholder="Rechercher des groupes" id="searchInput">
                <div class="icon filter-icon"></div>
            </div>
            <div class="actions">
                <button class="btn btn-primary" onclick="openCreateGroupModal()">
                    + Créer un groupe
                </button>
            </div>
        </header>

        <main class="main">
            <div class="page-header">
                <h2>👥 Mes Groupes</h2>
                <p><?php echo count($my_groups); ?> groupe(s) rejoint(s)</p>
            </div>

            <!-- Mes groupes -->
            <section class="my-groups">
                <h3>🏠 Mes groupes</h3>
                
                <?php if (empty($my_groups)): ?>
                <div class="empty-state">
                    <div class="empty-icon">👥</div>
                    <h4>Aucun groupe rejoint</h4>
                    <p>Créez votre premier groupe ou rejoignez-en un existant !</p>
                    <button class="btn btn-primary" onclick="openCreateGroupModal()">
                        Créer un groupe
                    </button>
                </div>
                <?php else: ?>
                
                <div class="groups-grid">
                    <?php foreach ($my_groups as $group): ?>
                    <div class="group-card">
                        <div class="group-header">
                            <h4><?php echo htmlspecialchars($group['name']); ?></h4>
                            <?php if ($group['owner_id'] == $_SESSION['user_id']): ?>
                            <span class="owner-badge">Propriétaire</span>
                            <?php endif; ?>
                        </div>
                        
                        <div class="group-description">
                            <?php echo htmlspecialchars($group['description']); ?>
                        </div>
                        
                        <div class="group-stats">
                            <div class="stat">
                                <span class="stat-number"><?php echo $group['member_count']; ?></span>
                                <span class="stat-label">membre(s)</span>
                            </div>
                            <div class="stat">
                                <span class="stat-label">Créé par</span>
                                <span class="stat-value"><?php echo htmlspecialchars($group['owner_name']); ?></span>
                            </div>
                        </div>
                        
                        <div class="group-actions">
                            <button class="btn btn-outline" onclick="viewGroup(<?php echo $group['id']; ?>)">
                                Voir le groupe
                            </button>
                            <?php if ($group['owner_id'] != $_SESSION['user_id']): ?>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="leave_group">
                                <input type="hidden" name="group_id" value="<?php echo $group['id']; ?>">
                                <button type="submit" class="btn btn-danger" onclick="return confirm('Quitter ce groupe ?')">
                                    Quitter
                                </button>
                            </form>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
                <?php endif; ?>
            </section>

            <!-- Groupes suggérés -->
            <?php if (!empty($public_groups)): ?>
            <section class="suggested-groups">
                <h3>💡 Groupes suggérés</h3>
                <div class="suggestions-grid">
                    <?php foreach ($public_groups as $group): ?>
                    <div class="suggestion-card">
                        <div class="suggestion-header">
                            <h4><?php echo htmlspecialchars($group['name']); ?></h4>
                            <span class="member-count"><?php echo $group['member_count']; ?> membre(s)</span>
                        </div>
                        
                        <div class="suggestion-description">
                            <?php echo htmlspecialchars(substr($group['description'], 0, 100)); ?>...
                        </div>
                        
                        <div class="suggestion-owner">
                            Créé par <?php echo htmlspecialchars($group['owner_name']); ?>
                        </div>
                        
                        <div class="suggestion-actions">
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="join_group">
                                <input type="hidden" name="group_id" value="<?php echo $group['id']; ?>">
                                <button type="submit" class="btn btn-primary">Rejoindre</button>
                            </form>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </section>
            <?php endif; ?>

            <!-- Activités récentes -->
            <?php if (!empty($group_activities)): ?>
            <section class="group-activities">
                <h3>🎭 Activités récentes</h3>
                <div class="activities-list">
                    <?php foreach ($group_activities as $activity): ?>
                    <div class="activity-item">
                        <div class="activity-content">
                            <strong><?php echo htmlspecialchars($activity['display_name']); ?></strong>
                            a rejoint le groupe
                            <strong><?php echo htmlspecialchars($activity['group_name']); ?></strong>
                        </div>
                        <div class="activity-date">
                            <?php echo date('d/m/Y à H:i', strtotime($activity['activity_date'])); ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </section>
            <?php endif; ?>
        </main>

        <aside class="aside">
            <div class="profile">
                <img src="assets/images/avatar-placeholder.jpg" alt="Avatar"/>
                <div class="info">
                    <div><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                    <div>@<?php echo strtolower(str_replace(' ', '', $_SESSION['user_name'])); ?></div>
                </div>
            </div>
            
            <div class="group-stats">
                <h4>📊 Vos statistiques</h4>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count($my_groups); ?></span>
                    <span class="stat-label">Groupes rejoints</span>
                </div>
                <div class="stat-item">
                    <span class="stat-number"><?php echo count(array_filter($my_groups, function($g) { return $g['owner_id'] == $_SESSION['user_id']; })); ?></span>
                    <span class="stat-label">Groupes créés</span>
                </div>
            </div>
            
            <div class="group-tips">
                <h4>💡 Conseils</h4>
                <ul>
                    <li>Créez des groupes thématiques</li>
                    <li>Invitez vos amis</li>
                    <li>Organisez des sorties ensemble</li>
                </ul>
            </div>
        </aside>
    </div>

    <!-- Modal de création de groupe -->
    <div id="createGroupModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Créer un nouveau groupe</h3>
                <button class="modal-close" onclick="closeCreateGroupModal()">&times;</button>
            </div>
            <form method="POST" class="modal-form">
                <input type="hidden" name="action" value="create_group">
                
                <div class="form-group">
                    <label for="group_name">Nom du groupe *</label>
                    <input type="text" id="group_name" name="name" required 
                           placeholder="Ex: Amateurs de comédies">
                </div>
                
                <div class="form-group">
                    <label for="group_description">Description</label>
                    <textarea id="group_description" name="description" rows="3" 
                              placeholder="Décrivez votre groupe et ses objectifs..."></textarea>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn btn-outline" onclick="closeCreateGroupModal()">
                        Annuler
                    </button>
                    <button type="submit" class="btn btn-primary">
                        Créer le groupe
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script src="assets/js/groups.js"></script>
</body>
</html>
